<?php
/**
 * PvP Rankings
 * https://webenginecms.org/
 * 
 * @version 1.1.0
 * @author Lautaro Angelico <http://lautaroangelico.com/>
 * @copyright (c) 2013-2018 Lautaro Angelico, All Rights Reserved
 * @build w3c8c718b75a0f1fa1a557f7f9d70877
 */

namespace Plugin\PVPRankings;

class PVPRankings {
	
	private $_configFile = 'config.json';
	private $_modulesPath = 'modules';
	private $_cacheFile = 'rankings_pvp.cache';
	
	private $_initialScore;
	private $_matchScoreCap;
	private $_matchLowScoreCap;
	private $_rankImageSize;
	private $_specialRanks;
	private $_ranks;
	
	// CONSTRUCTOR
	
	function __construct() {
		
		// load configs
		$this->configFilePath = __PATH_PVPRANKINGS_ROOT__.$this->_configFile;
		if(!file_exists($this->configFilePath)) throw new \Exception(lang('pvprankings_error_2', true));
		$cfg = file_get_contents($this->configFilePath);
		if(!check_value($cfg)) throw new \Exception(lang('pvprankings_error_2', true));
		$cfg = json_decode($cfg, true);
		if(!is_array($cfg)) throw new \Exception(lang('pvprankings_error_2', true));
		
		// set configs
		$this->_initialScore = $cfg['initial_score'];
		$this->_matchScoreCap = $cfg['match_score_cap'];
		$this->_matchLowScoreCap = $cfg['match_low_score_cap'];
		$this->_rankImageSize = $cfg['rank_image_size'];
		$this->_specialRanks = $cfg['special_ranks'];
		$this->_ranks = $cfg['ranks'];
		
		// load database
		$this->db = \Connection::Database('MuOnline');
		$this->db2 = \Connection::Database('Me_MuOnline');
		
		// cron
		$this->_checkCron();
	}
	
	// PUBLIC FUNCTIONS
	
	public function loadModule($module) {
		if(!\Validator::Alpha($module)) throw new \Exception(lang('pvprankings_error_4', true));
		if(!$this->_moduleExists($module)) throw new \Exception(lang('pvprankings_error_4', true));
		if(!@include_once(__PATH_PVPRANKINGS_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) throw new \Exception(lang('pvprankings_error_4', true));
	}
	
	public function updateCache() {
		
		$killsList = $this->db->query_fetch("SELECT * FROM C_PlayerKiller_Info ORDER BY id ASC");
		$scoreList = array();
		$finalScoreList = array();
		
		// Initial score list
		foreach($killsList as $row) {
			
			if(!array_key_exists($row['Victim'], $scoreList)) {
				$scoreList[$row['Victim']] = $this->_initialScore;
			}
			
			if(!array_key_exists($row['Killer'], $scoreList)) {
				$scoreList[$row['Killer']] = $this->_initialScore;
			}
			
		}

		// calculate win/lose
		foreach($killsList as $row2) {
			
			$killerScore = $scoreList[$row2['Killer']];
			$victimScore = $scoreList[$row2['Victim']];
			
			// la formula secreta
			$matchScore = abs(round(25*($scoreList[$row2['Victim']]/$scoreList[$row2['Killer']])));
			
			// match score cap
			if($this->_matchScoreCap > 0) {
				if($matchScore > $this->_matchScoreCap) $matchScore = $this->_matchScoreCap;
			}
			
			// match low score cap
			if($matchScore < $this->_matchLowScoreCap) {
				$matchScore = 0;
			}
			
			$scoreList[$row2['Killer']] += $matchScore;
			$scoreList[$row2['Victim']] -= $matchScore;
			
			
		}

		// Order list
		arsort($scoreList);

		// Prepare final array
		foreach($scoreList as $playerName => $playerScore) {
			$finalScoreList[] = array($playerName, $playerScore, $this->_getRankGraphicBasedOnScore($playerScore), $this->_getRankBasedOnScore($playerScore));
		}

		// Special Ranks (graphics)
		$finalScoreList[0][2] = $this->_specialRanks[0][1];
		$finalScoreList[1][2] = $this->_specialRanks[1][1];
		$finalScoreList[2][2] = $this->_specialRanks[1][1];
		$finalScoreList[3][2] = $this->_specialRanks[1][1];
		$finalScoreList[4][2] = $this->_specialRanks[1][1];
		$finalScoreList[5][2] = $this->_specialRanks[2][1];
		$finalScoreList[6][2] = $this->_specialRanks[2][1];
		$finalScoreList[7][2] = $this->_specialRanks[2][1];
		$finalScoreList[8][2] = $this->_specialRanks[2][1];
		$finalScoreList[9][2] = $this->_specialRanks[2][1];
		$finalScoreList[10][2] = $this->_specialRanks[3][1];
		$finalScoreList[11][2] = $this->_specialRanks[3][1];
		$finalScoreList[12][2] = $this->_specialRanks[3][1];
		$finalScoreList[13][2] = $this->_specialRanks[3][1];
		$finalScoreList[14][2] = $this->_specialRanks[3][1];
		$finalScoreList[15][2] = $this->_specialRanks[3][1];
		$finalScoreList[16][2] = $this->_specialRanks[3][1];
		$finalScoreList[17][2] = $this->_specialRanks[3][1];
		$finalScoreList[18][2] = $this->_specialRanks[3][1];
		$finalScoreList[19][2] = $this->_specialRanks[3][1];

		// Special Ranks (titles)
		$finalScoreList[0][3] = $this->_specialRanks[0][0];
		$finalScoreList[1][3] = $this->_specialRanks[1][0];
		$finalScoreList[2][3] = $this->_specialRanks[1][0];
		$finalScoreList[3][3] = $this->_specialRanks[1][0];
		$finalScoreList[4][3] = $this->_specialRanks[1][0];
		$finalScoreList[5][3] = $this->_specialRanks[2][0];
		$finalScoreList[6][3] = $this->_specialRanks[2][0];
		$finalScoreList[7][3] = $this->_specialRanks[2][0];
		$finalScoreList[8][3] = $this->_specialRanks[2][0];
		$finalScoreList[9][3] = $this->_specialRanks[2][0];
		$finalScoreList[10][3] = $this->_specialRanks[3][0];
		$finalScoreList[11][3] = $this->_specialRanks[3][0];
		$finalScoreList[12][3] = $this->_specialRanks[3][0];
		$finalScoreList[13][3] = $this->_specialRanks[3][0];
		$finalScoreList[14][3] = $this->_specialRanks[3][0];
		$finalScoreList[15][3] = $this->_specialRanks[3][0];
		$finalScoreList[16][3] = $this->_specialRanks[3][0];
		$finalScoreList[17][3] = $this->_specialRanks[3][0];
		$finalScoreList[18][3] = $this->_specialRanks[3][0];
		$finalScoreList[19][3] = $this->_specialRanks[3][0];
		
		if(!is_array($finalScoreList)) return;
		
		$cacheData = $this->_prepareCacheData($finalScoreList);
		if(!$this->_updateCacheFile($cacheData)) return;
		return true;
		
	}
	
	public function formatRankBadge($file) {
		return '<img src="'.__PATH_TEMPLATE_IMG__.'ranks/'.$file.'" width="'.$this->_rankImageSize.'px" height="auto">';
	}
	
	public function loadCacheData() {
		$file = __PATH_CACHE__.$this->_cacheFile;
		if(!file_exists($file)) return;
		
		$cacheData = file_get_contents($file);
		if(!check_value($cacheData)) return;
		
		$cacheDataArray = json_decode($cacheData, true);
		if(!is_array($cacheDataArray)) return;
		
		return $cacheDataArray;
	}
	
	// PRIVATE FUNCTIONS
	
	private function _moduleExists($module) {
		if(!check_value($module)) return;
		if(!file_exists(__PATH_PVPRANKINGS_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) return;
		return true;
	}
	
	private function _getRankBasedOnScore($score) {
		if($score<1) return '';
		foreach($this->_ranks as $rankScore => $rankData) {
			if($score > $rankScore) {
				$rank = $rankData[0];
			}
		}
		return $rank;
	}

	private function _getRankGraphicBasedOnScore($score) {
		if($score<1) return '';
		foreach($this->_ranks as $rankScore => $rankData) {
			if($score > $rankScore) {
				$rank = $rankData[1];
			}
		}
		return $rank;
	}

	private function _prepareCacheData($data) {
		if(!is_array($data)) return;
		return json_encode($data);
	}
	
	private function _updateCacheFile($data) {
		$file = __PATH_CACHE__.$this->_cacheFile;
		if(!file_exists($file)) return;
		if(!is_writable($file)) return;
		
		$fp = fopen($file, 'w');
		fwrite($fp, $data);
		fclose($fp);
		return true;
	}
	
	private function _checkCron() {
		$result = $this->db2->query_fetch_single("SELECT * FROM ".WEBENGINE_CRON." WHERE cron_file_run = ?", array('pvp_rankings.php'));
		if(is_array($result)) return;
		$this->_createCron();
	}
	
	private function _createCron() {
		if(!file_exists(__PATH_CRON__ . 'pvp_rankings.php')) throw new \Exception(lang('pvprankings_error_3', true));
		$cronMd5 = md5_file(__PATH_CRON__ . 'pvp_rankings.php');
		if(!check_value($cronMd5)) throw new \Exception(lang('pvprankings_error_3', true));
		$insertData = array(
			'PvP Rankings',
			'pvp_rankings.php',
			3600,
			1,
			0,
			$cronMd5
		);
		$result = $this->db2->query("INSERT INTO ".WEBENGINE_CRON." (cron_name, cron_file_run, cron_run_time, cron_status, cron_protected, cron_file_md5) VALUES (?, ?, ?, ?, ?, ?)", $insertData);
		if(!$result) throw new \Exception(lang('pvprankings_error_3', true));
	}
	
}